const fs = require('fs');
const path = require('path');
const logger = require('./logger');

const videoExtensions = ['mp4', 'avi', 'mov', 'mkv', 'wmv', 'flv', 'webm', 'm4v', 'mpg', 'mpeg', '3gp'];

function scanDirectory(dirPath) {
  const videos = [];

  function scanDir(currentPath, relativePath = '') {
    try {
      const entries = fs.readdirSync(currentPath, { withFileTypes: true });

      for (const entry of entries) {
        const fullPath = path.join(currentPath, entry.name);
        const relPath = path.join(relativePath, entry.name);

        try {
          if (entry.isDirectory()) {
            scanDir(fullPath, relPath);
          } else if (entry.isFile()) {
            const ext = path.extname(entry.name).toLowerCase().slice(1);
            if (videoExtensions.includes(ext)) {
              const stats = fs.statSync(fullPath);
              videos.push({
                path: fullPath,
                name: entry.name,
                relativePath: relPath,
                directory: relativePath ? path.dirname(relPath) : '',
                size: stats.size,
                created: stats.birthtime,
                modified: stats.mtime,
                extension: ext
              });
            }
          }
        } catch (err) {
          continue;
        }
      }
    } catch (error) {
      logger.error(`Error scanning directory ${currentPath}:`, error);
    }
  }

  scanDir(dirPath);
  return videos;
}

function buildDirectoryStructure(directories) {
  const structure = [];

  directories.forEach((dirObj) => {
    const dirPath = typeof dirObj === 'string' ? dirObj : dirObj.path;
    const dirName = typeof dirObj === 'string' ? path.basename(dirPath) : dirObj.name;

    if (!fs.existsSync(dirPath)) {
      return;
    }

    const tree = {
      name: dirName,
      path: dirPath,
      children: []
    };

    function buildTree(currentPath, parentNode) {
      try {
        const entries = fs.readdirSync(currentPath, { withFileTypes: true });

        for (const entry of entries) {
          const fullPath = path.join(currentPath, entry.name);

          try {
            if (entry.isDirectory()) {
              const node = {
                name: entry.name,
                path: fullPath,
                children: []
              };
              parentNode.children.push(node);
              buildTree(fullPath, node);
            }
          } catch (err) {
            continue;
          }
        }
      } catch (error) {
        logger.error(`Error building tree for ${currentPath}:`, error);
      }
    }

    buildTree(dirPath, tree);
    structure.push(tree);
  });

  return structure;
}

module.exports = {
  videoExtensions,
  scanDirectory,
  buildDirectoryStructure
};

